/*
 Copyright (c) 2011, Tony Million.
 All rights reserved.
 
 Redistribution and use in source and binary forms, with or without
 modification, are permitted provided that the following conditions are met:
 
 1. Redistributions of source code must retain the above copyright notice, this
 list of conditions and the following disclaimer.
 
 2. Redistributions in binary form must reproduce the above copyright notice,
 this list of conditions and the following disclaimer in the documentation
 and/or other materials provided with the distribution.
 
 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 POSSIBILITY OF SUCH DAMAGE. 
 */
import UIKit
import SystemConfiguration

let NEEDS_DISPATCH_RETAIN_RELEASE = 0

let kReachabilityChangedNotification = ""

enum NetworkStatus : Int {
    // Apple NetworkStatus Compatible Names.
    case NotReachable = 0
    case ReachableViaWiFi = 2
    case ReachableViaWWAN = 1
}

typealias NetworkReachable = (reachability: Reachability) -> Void
typealias NetworkUnreachable = (reachability: Reachability) -> Void
class Reachability: NSObject {
    var reachableBlock = NetworkReachable()
    var unreachableBlock = NetworkUnreachable()
    var reachableOnWWAN = false

    class func reachabilityWithHostname(hostname: String) -> Reachability {
        var ref = SCNetworkReachabilityCreateWithName(nil, hostname.UTF8String)
        if ref {
            var reachability = self.init(reachabilityRef: ref)
            return reachability
        }
        return nil
    }

    class func reachabilityForInternetConnection() -> Reachability {
        var zeroAddress: structsockaddr_in
        bzero(zeroAddress, sizeof(zeroAddress))
        zeroAddress.sin_len = sizeof(zeroAddress)
        zeroAddress.sin_family = AF_INET
        return self.reachabilityWithAddress(zeroAddress)
    }

    class func reachabilityWithAddress(hostAddress: structsockaddr_in) -> Reachability {
        var ref = SCNetworkReachabilityCreateWithAddress(kCFAllocatorDefault, (hostAddress as! structsockaddr))
        if ref {
            var reachability = self.init(reachabilityRef: ref)
            return reachability
        }
        return nil
    }

    class func reachabilityForLocalWiFi() -> Reachability {
        var localWifiAddress: structsockaddr_in
        bzero(localWifiAddress, sizeof(localWifiAddress))
        localWifiAddress.sin_len = sizeof(localWifiAddress)
        localWifiAddress.sin_family = AF_INET
        // IN_LINKLOCALNETNUM is defined in <netinet/in.h> as 169.254.0.0
        localWifiAddress.sin_addr.s_addr = htonl(IN_LINKLOCALNETNUM)
        return self.reachabilityWithAddress(localWifiAddress)
    }

    override func initWithReachabilityRef(ref: SCNetworkReachabilityRef) -> Reachability {
        super.init()
        if self != nil {
            self.reachableOnWWAN = true
            self.reachabilityRef = ref
        }
    }

    func startNotifier() -> Bool {
        var context = [0, nil, nil, nil, nil]
        // this should do a retain on ourself, so as long as we're in notifier mode we shouldn't disappear out from under ourselves
        // woah
        self.reachabilityObject = self
        // first we need to create a serial queue
        // we allocate this once for the lifetime of the notifier
        self.reachabilitySerialQueue = dispatch_queue_create("com.tonymillion.reachability")
        if !self.reachabilitySerialQueue {
            return false
        }
        context.info = (self as! Void)
        if !SCNetworkReachabilitySetCallback(self.reachabilityRef, TMReachabilityCallback, context) {
//#if DEBUG
//            print("SCNetworkReachabilitySetCallback() failed: \(SCErrorString(SCError()))")
//#endif
            //clear out the dispatch queue
            if self.reachabilitySerialQueue {
                self.reachabilitySerialQueue = nil
            }
            self.reachabilityObject = nil
            return false
        }
        // set it as our reachability queue which will retain the queue
        if !SCNetworkReachabilitySetDispatchQueue(self.reachabilityRef, self.reachabilitySerialQueue) {
            //UH OH - FAILURE!
            // first stop any callbacks!
            SCNetworkReachabilitySetCallback(self.reachabilityRef, nil, nil)
            // then clear out the dispatch queue
            if self.reachabilitySerialQueue {
                self.reachabilitySerialQueue = nil
            }
            self.reachabilityObject = nil
            return false
        }
        return true
    }

    func stopNotifier() {
        // first stop any callbacks!
        SCNetworkReachabilitySetCallback(self.reachabilityRef, nil, nil)
        // unregister target from the GCD serial dispatch queue
        SCNetworkReachabilitySetDispatchQueue(self.reachabilityRef, nil)
        if self.reachabilitySerialQueue {
            self.reachabilitySerialQueue = nil
        }
        self.reachabilityObject = nil
    }

    func isReachable() -> Bool {
        var flags: SCNetworkReachabilityFlags
        if !SCNetworkReachabilityGetFlags(self.reachabilityRef, flags) {
            return false
        }
        return self.isReachableWithFlags(flags)
    }

    func isReachableViaWWAN() -> Bool {
        var flags = 0
        if SCNetworkReachabilityGetFlags(self.reachabilityRef, flags) {
            // check we're REACHABLE
            if flags & kSCNetworkReachabilityFlagsReachable {
                // now, check we're on WWAN
                if flags & kSCNetworkReachabilityFlagsIsWWAN {
                    return true
                }
            }
        }
        return false
    }

    func isReachableViaWiFi() -> Bool {
        var flags = 0
        if SCNetworkReachabilityGetFlags(self.reachabilityRef, flags) {
            // check we're reachable
            if (flags & kSCNetworkReachabilityFlagsReachable) {
                // check we're NOT on WWAN
                if (flags & kSCNetworkReachabilityFlagsIsWWAN) {
                    return false
                }
                return true
            }
        }
        return false
    }
    // WWAN may be available, but not active until a connection has been established.
    // WiFi may require a connection for VPN on Demand.

    func isConnectionRequired() -> Bool {
        return self.connectionRequired()
    }
    // Identical DDG variant.

    func connectionRequired() -> Bool {
        var flags: SCNetworkReachabilityFlags
        if SCNetworkReachabilityGetFlags(self.reachabilityRef, flags) {
            return (flags & kSCNetworkReachabilityFlagsConnectionRequired)
        }
        return false
    }
    // Apple's routine.
    // Dynamic, on demand connection?

    func isConnectionOnDemand() -> Bool {
        var flags: SCNetworkReachabilityFlags
        if SCNetworkReachabilityGetFlags(self.reachabilityRef, flags) {
            return (flags & kSCNetworkReachabilityFlagsConnectionRequired) && (flags & (kSCNetworkReachabilityFlagsConnectionOnTraffic | kSCNetworkReachabilityFlagsConnectionOnDemand))
        }
        return false
    }
    // Is user intervention required?

    func isInterventionRequired() -> Bool {
        var flags: SCNetworkReachabilityFlags
        if SCNetworkReachabilityGetFlags(self.reachabilityRef, flags) {
            return (flags & kSCNetworkReachabilityFlagsConnectionRequired) && (flags & kSCNetworkReachabilityFlagsInterventionRequired)
        }
        return false
    }

    func currentReachabilityStatus() -> NetworkStatus {
        if self.isReachable() {
            if self.isReachableViaWiFi() {
                return ReachableViaWiFi
            }
            return ReachableViaWWAN
        }
        return NotReachable
    }

    func reachabilityFlags() -> SCNetworkReachabilityFlags {
        var flags = 0
        if SCNetworkReachabilityGetFlags(self.reachabilityRef, flags) {
            return flags
        }
        return 0
    }

    func currentReachabilityString() -> String {
        var temp = self.currentReachabilityStatus()
        if temp == self.reachableOnWWAN {
            // updated for the fact we have CDMA phones now!
            return NSLocalizedString("Cellular", "")
        }
        if temp == ReachableViaWiFi {
            return NSLocalizedString("WiFi", "")
        }
        return NSLocalizedString("No Connection", "")
    }

    func currentReachabilityFlags() -> String {
        return reachabilityFlags(self.reachabilityFlags())
    }

// MARK: - class constructor methods
    // initialization methods

    deinit {
        self.stopNotifier()
        if self.reachabilityRef {
            CFRelease(self.reachabilityRef)
            self.reachabilityRef = nil
        }
        super.dealloc()
    }
// MARK: - notifier methods
    // Notifier 
    // NOTE: this uses GCD to trigger the blocks - they *WILL NOT* be called on THE MAIN THREAD
    // - In other words DO NOT DO ANY UI UPDATES IN THE BLOCKS.
    //   INSTEAD USE dispatch_async(dispatch_get_main_queue(), ^{UISTUFF}) (or dispatch_sync if you want)
// MARK: - reachability tests
    // this is for the case where you flick the airplane mode
    // you end up getting something like this:
    //Reachability: WR ct-----
    //Reachability: -- -------
    //Reachability: WR ct-----
    //Reachability: -- -------
    // we treat this as 4 UNREACHABLE triggers - really apple should do better than this
let testcase = (kSCNetworkReachabilityFlagsConnectionRequired | kSCNetworkReachabilityFlagsTransientConnection)

    func isReachableWithFlags(flags: SCNetworkReachabilityFlags) -> Bool {
        var connectionUP = true
        if !(flags & kSCNetworkReachabilityFlagsReachable) {
            connectionUP = false
        }
        if (flags & testcase) == testcase {
            connectionUP = false
        }
        if flags & kSCNetworkReachabilityFlagsIsWWAN {
            // we're on 3G
            if !self.reachableOnWWAN {
                // we dont want to connect when on 3G
                connectionUP = false
            }
        }
        return connectionUP
    }
    // WWAN may be available, but not active until a connection has been established.
    // WiFi may require a connection for VPN on Demand.
    // Dynamic, on demand connection?
    // Is user intervention required?
// MARK: - reachability status stuff
// MARK: - callback function calls this method

    func reachabilityChanged(flags: SCNetworkReachabilityFlags) {
        if self.isReachableWithFlags(flags) {
            if self.reachableBlock {
                self.reachableBlock(self)
            }
        }
        else {
            if self.unreachableBlock {
                self.unreachableBlock(self)
            }
        }
        // this makes sure the change notification happens on the MAIN THREAD
        dispatch_async(dispatch_get_main_queue(), {() -> Void in
            NSNotificationCenter.defaultCenter().postNotificationName(kReachabilityChangedNotification, object: self)
        })
    }

    var reachabilityRef = SCNetworkReachabilityRef()
    var reachabilitySerialQueue = dispatch_queue_t()
    var reachabilityObject: AnyObject!

    func reachabilityChanged(flags: SCNetworkReachabilityFlags) {
    }

    func isReachableWithFlags(flags: SCNetworkReachabilityFlags) -> Bool {
    }
}
/*
 Copyright (c) 2011, Tony Million.
 All rights reserved.
 
 Redistribution and use in source and binary forms, with or without
 modification, are permitted provided that the following conditions are met:
 
 1. Redistributions of source code must retain the above copyright notice, this
 list of conditions and the following disclaimer.
 
 2. Redistributions in binary form must reproduce the above copyright notice,
 this list of conditions and the following disclaimer in the documentation
 and/or other materials provided with the distribution.
 
 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 POSSIBILITY OF SUCH DAMAGE. 
 */
let kReachabilityChangedNotification = "kReachabilityChangedNotification"

func reachabilityFlags(flags: SCNetworkReachabilityFlags) -> String {
    return "\((flags & kSCNetworkReachabilityFlagsIsWWAN) ? "W" : "-")\("X")\((flags & kSCNetworkReachabilityFlagsReachable) ? "R" : "-")\((flags & kSCNetworkReachabilityFlagsConnectionRequired) ? "c" : "-")\((flags & kSCNetworkReachabilityFlagsTransientConnection) ? "t" : "-")\((flags & kSCNetworkReachabilityFlagsInterventionRequired) ? "i" : "-")\((flags & kSCNetworkReachabilityFlagsConnectionOnTraffic) ? "C" : "-")\((flags & kSCNetworkReachabilityFlagsConnectionOnDemand) ? "D" : "-")\((flags & kSCNetworkReachabilityFlagsIsLocalAddress) ? "l" : "-")"
}

//Start listening for reachability notifications on the current run loop
func TMReachabilityCallback(target: SCNetworkReachabilityRef, flags: SCNetworkReachabilityFlags, info: Void) {
//#pragma unused (target)
    var reachability = (info as! Reachability)
    // we probably dont need an autoreleasepool here as GCD docs state each queue has its own autorelease pool
    // but what the heck eh?
            reachability.reachabilityChanged(flags)

}